<?php

namespace App\Controllers;

use App\Models\OrderModel;
use App\Models\UserModel;
use App\Models\NotificationModel;
use App\Models\OrderStatusHistoryModel;

class Delivery extends BaseController
{
    protected $orderModel;
    protected $userModel;
    protected $notificationModel;
    protected $statusHistoryModel;

    public function __construct()
    {
        $this->orderModel = new OrderModel();
        $this->userModel = new UserModel();
        $this->notificationModel = new NotificationModel();
        $this->statusHistoryModel = new OrderStatusHistoryModel();
    }

    // D1 - Dashboard
    public function dashboard()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'delivery') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        // Get assigned orders
        $assignedOrders = $this->orderModel
            ->where('delivery_person_id', $userId)
            ->where('status !=', 'completed')
            ->where('status !=', 'cancelled')
            ->orderBy('created_at', 'DESC')
            ->findAll();

        // Get completed deliveries
        $completedOrders = $this->orderModel
            ->where('delivery_person_id', $userId)
            ->whereIn('status', ['completed', 'returned'])
            ->orderBy('updated_at', 'DESC')
            ->limit(10)
            ->findAll();

        // Get statistics
        $stats = [
            'pending' => $this->orderModel->where('delivery_person_id', $userId)->where('status', 'dispatched')->countAllResults(),
            'in_transit' => $this->orderModel->where('delivery_person_id', $userId)->where('status', 'in_transit')->countAllResults(),
            'delivered' => $this->orderModel->where('delivery_person_id', $userId)->where('status', 'delivered')->countAllResults(),
            'total_completed' => $this->orderModel->where('delivery_person_id', $userId)->where('status', 'completed')->countAllResults(),
        ];

        $data = [
            'user' => $user,
            'assignedOrders' => $assignedOrders,
            'completedOrders' => $completedOrders,
            'stats' => $stats,
            'notifications' => $this->notificationModel->getUserNotifications($userId, true),
        ];

        return view('delivery/dashboard', $data);
    }

    // D2 - View Order Details
    public function orderDetails($orderId)
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'delivery') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $order = $this->orderModel->find($orderId);

        if (!$order || $order['delivery_person_id'] != $userId) {
            return redirect()->to('/delivery/dashboard')->with('error', 'Order not found');
        }

        // Get order history
        $history = $this->statusHistoryModel->getOrderHistory($orderId);

        // Get buyer and seller details
        $buyer = $this->userModel->find($order['buyer_id']);
        $seller = $this->userModel->find($order['seller_id']);

        $data = [
            'user' => $this->userModel->find($userId),
            'order' => $order,
            'buyer' => $buyer,
            'seller' => $seller,
            'history' => $history,
        ];

        return view('delivery/order_details', $data);
    }

    // D3 - Update Pickup Status
    public function markPickedUp($orderId)
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'delivery') {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $userId = session()->get('user_id');
        $order = $this->orderModel->find($orderId);

        if (!$order || $order['delivery_person_id'] != $userId) {
            return $this->response->setJSON(['success' => false, 'message' => 'Order not found']);
        }

        if ($order['status'] !== 'dispatched') {
            return $this->response->setJSON(['success' => false, 'message' => 'Order cannot be picked up at this stage']);
        }

        // Update order status to in_transit
        $this->orderModel->updateOrderStatus($orderId, 'in_transit', $userId, 'Order picked up by delivery person');

        // Notify buyer
        $this->notificationModel->insert([
            'user_id' => $order['buyer_id'],
            'title' => 'Order Picked Up',
            'message' => 'Your order has been picked up and is on the way',
            'type' => 'order_update',
            'related_id' => $orderId,
        ]);

        return $this->response->setJSON(['success' => true, 'message' => 'Order marked as picked up']);
    }

    // D4 - Mark as Delivered
    public function markDelivered($orderId)
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'delivery') {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $userId = session()->get('user_id');
        $order = $this->orderModel->find($orderId);

        if (!$order || $order['delivery_person_id'] != $userId) {
            return $this->response->setJSON(['success' => false, 'message' => 'Order not found']);
        }

        if ($order['status'] !== 'in_transit') {
            return $this->response->setJSON(['success' => false, 'message' => 'Order not ready for delivery']);
        }

        $otp = $this->request->getPost('delivery_otp');
        
        // Validate OTP (assume OTP is stored in order or sent to buyer)
        // For now, we'll skip OTP validation - add later if needed

        // Update order status
        $this->orderModel->update($orderId, [
            'status' => 'delivered',
            'delivery_date' => date('Y-m-d H:i:s'),
        ]);

        $this->statusHistoryModel->insert([
            'order_id' => $orderId,
            'status' => 'delivered',
            'notes' => 'Order delivered successfully',
            'updated_by' => $userId,
        ]);

        // Notify buyer and seller
        $this->notificationModel->insert([
            'user_id' => $order['buyer_id'],
            'title' => 'Order Delivered',
            'message' => 'Your order has been delivered successfully',
            'type' => 'order_delivered',
            'related_id' => $orderId,
        ]);

        $this->notificationModel->insert([
            'user_id' => $order['seller_id'],
            'title' => 'Order Delivered',
            'message' => 'Your product has been delivered to the buyer',
            'type' => 'order_delivered',
            'related_id' => $orderId,
        ]);

        return $this->response->setJSON(['success' => true, 'message' => 'Order marked as delivered']);
    }

    // D5 - Delivery History
    public function history()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'delivery') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        $deliveries = $this->orderModel
            ->where('delivery_person_id', $userId)
            ->orderBy('updated_at', 'DESC')
            ->findAll();

        $data = [
            'user' => $user,
            'deliveries' => $deliveries,
        ];

        return view('delivery/history', $data);
    }

    // D6 - Earnings Summary
    public function earnings()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'delivery') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        // Calculate earnings (assuming delivery charge per order)
        $completedDeliveries = $this->orderModel
            ->where('delivery_person_id', $userId)
            ->where('status', 'completed')
            ->findAll();

        $totalEarnings = count($completedDeliveries) * 50; // ₹50 per delivery (adjust as needed)

        $data = [
            'user' => $user,
            'totalEarnings' => $totalEarnings,
            'completedDeliveries' => $completedDeliveries,
        ];

        return view('delivery/earnings', $data);
    }
}
