<?php

namespace App\Controllers;

use App\Models\UserModel;
use App\Models\ProductModel;
use App\Models\OrderModel;
use App\Models\OfferModel;
use App\Models\NotificationModel;

class SuperAdmin extends BaseController
{
    protected $userModel;
    protected $productModel;
    protected $orderModel;
    protected $offerModel;
    protected $notificationModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->productModel = new ProductModel();
        $this->orderModel = new OrderModel();
        $this->offerModel = new OfferModel();
        $this->notificationModel = new NotificationModel();
    }

    // SA1 - Dashboard with system overview
    public function dashboard()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        // Get system statistics
        $stats = [
            'total_users' => $this->userModel->countAllResults(),
            'total_buyers' => $this->userModel->where('user_type', 'buyer')->countAllResults(),
            'total_sellers' => $this->userModel->where('user_type', 'seller')->countAllResults(),
            'total_delivery' => $this->userModel->where('user_type', 'delivery')->countAllResults(),
            'total_admins' => $this->userModel->where('user_type', 'admin')->countAllResults(),
            'total_products' => $this->productModel->countAllResults(),
            'pending_products' => $this->productModel->where('status', 'pending')->countAllResults(),
            'approved_products' => $this->productModel->where('status', 'approved')->countAllResults(),
            'total_orders' => $this->orderModel->countAllResults(),
            'active_orders' => $this->orderModel->whereIn('status', ['pending', 'dispatched', 'in_transit'])->countAllResults(),
        ];

        // Get recent activities
        $recentUsers = $this->userModel->orderBy('created_at', 'DESC')->limit(10)->findAll();
        $recentOrders = $this->orderModel->orderBy('created_at', 'DESC')->limit(10)->findAll();

        $data = [
            'user' => $user,
            'stats' => $stats,
            'recentUsers' => $recentUsers,
            'recentOrders' => $recentOrders,
        ];

        return view('superadmin/dashboard', $data);
    }

    // SA2 - Manage Admins
    public function admins()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        $admins = $this->userModel
            ->where('user_type', 'admin')
            ->orderBy('created_at', 'DESC')
            ->findAll();

        $data = [
            'user' => $user,
            'admins' => $admins,
        ];

        return view('superadmin/admins', $data);
    }

    // SA3 - Create Admin
    public function createAdmin()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $validation = \Config\Services::validation();
        
        $rules = [
            'name' => 'required|min_length[3]',
            'email' => 'required|valid_email|is_unique[users.email]',
            'mobile' => 'required|min_length[10]|max_length[15]',
            'password' => 'required|min_length[6]',
        ];

        if (!$this->validate($rules)) {
            return $this->response->setJSON([
                'success' => false,
                'errors' => $validation->getErrors()
            ]);
        }

        $adminData = [
            'name' => $this->request->getPost('name'),
            'email' => $this->request->getPost('email'),
            'mobile' => $this->request->getPost('mobile'),
            'password' => password_hash($this->request->getPost('password'), PASSWORD_DEFAULT),
            'user_type' => 'admin',
            'is_verified' => 1,
        ];

        $adminId = $this->userModel->insert($adminData);

        if ($adminId) {
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Admin created successfully'
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'Failed to create admin'
        ]);
    }

    // SA4 - Delete/Deactivate Admin
    public function deleteAdmin($adminId)
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $admin = $this->userModel->find($adminId);

        if (!$admin || $admin['user_type'] !== 'admin') {
            return $this->response->setJSON(['success' => false, 'message' => 'Admin not found']);
        }

        // Soft delete or deactivate
        $this->userModel->update($adminId, ['is_verified' => 0]);

        return $this->response->setJSON(['success' => true, 'message' => 'Admin deactivated successfully']);
    }

    // SA5 - System Settings
    public function settings()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        // Get system settings from database or config
        $settings = [
            'site_name' => 'Flex Market',
            'commission_rate' => 5, // %
            'delivery_charge' => 50,
            'min_order_value' => 100,
        ];

        $data = [
            'user' => $user,
            'settings' => $settings,
        ];

        return view('superadmin/settings', $data);
    }

    // SA6 - Update System Settings
    public function updateSettings()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        // Update settings in database or config
        // For now, just return success
        
        return $this->response->setJSON(['success' => true, 'message' => 'Settings updated successfully']);
    }

    // SA7 - View All Users
    public function users()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        $userType = $this->request->getGet('type');
        
        $builder = $this->userModel;
        if ($userType) {
            $builder->where('user_type', $userType);
        }

        $users = $builder->orderBy('created_at', 'DESC')->findAll();

        $data = [
            'user' => $user,
            'users' => $users,
            'filterType' => $userType,
        ];

        return view('superadmin/users', $data);
    }

    // SA8 - Suspend/Activate User
    public function toggleUserStatus($targetUserId)
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $targetUser = $this->userModel->find($targetUserId);

        if (!$targetUser) {
            return $this->response->setJSON(['success' => false, 'message' => 'User not found']);
        }

        // Toggle verified status (1 = active, 0 = suspended)
        $newStatus = $targetUser['is_verified'] ? 0 : 1;
        $this->userModel->update($targetUserId, ['is_verified' => $newStatus]);

        $message = $newStatus ? 'User activated successfully' : 'User suspended successfully';
        
        return $this->response->setJSON(['success' => true, 'message' => $message]);
    }

    // SA9 - System Reports
    public function reports()
    {
        if (!session()->get('logged_in') || session()->get('user_type') !== 'super_admin') {
            return redirect()->to('/login');
        }

        $userId = session()->get('user_id');
        $user = $this->userModel->find($userId);

        // Generate various reports
        $reports = [
            'daily_orders' => $this->orderModel->where('DATE(created_at)', date('Y-m-d'))->countAllResults(),
            'weekly_orders' => $this->orderModel->where('created_at >=', date('Y-m-d', strtotime('-7 days')))->countAllResults(),
            'monthly_orders' => $this->orderModel->where('created_at >=', date('Y-m-d', strtotime('-30 days')))->countAllResults(),
            'total_revenue' => 0, // Calculate from orders
        ];

        $data = [
            'user' => $user,
            'reports' => $reports,
        ];

        return view('superadmin/reports', $data);
    }
}
